﻿using System;
using Hims.Shared.UserModels.Common;
using Hims.Shared.UserModels.Package;
using Product = Hims.Shared.UserModels.Package.Product;
using Surgery = Hims.Shared.UserModels.Package.Surgery;
using MetaData = Hims.Shared.UserModels.Package.MetaData;

namespace Hims.Api.Controllers
{
    using System.Threading.Tasks;

    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Shared.EntityModels;
    using Hims.Shared.Library.Enums;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/package")]
    public class PackageController : BaseController
    {
        /// <summary>
        /// The service
        /// </summary>
        private readonly IPackageService service;

        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public PackageController(IPackageService service, IAuditLogService auditLogServices)
        {
            this.service = service;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The add icons async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch-meta-data")]
        public async Task<ActionResult> FetchMetaDataAsync([FromBody] MetaData.FilterModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model.LocationId = Convert.ToInt32(header.LocationId);
                var response = await this.service.FetchMetaDataAsync(model.Id,model.LocationId).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The add icons async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchAsync([FromBody] FilterModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model.LocationId = Convert.ToInt32(header.LocationId);
                var response = await this.service.FetchAsync(model).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }
        
        /// <summary>
        /// The add icons async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("insert")]
        public async Task<ActionResult> InsertAsync([FromBody] InsertModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model.LocationId = Convert.ToInt32(header.LocationId);
                var response = await this.service.InsertAsync(model).ConfigureAwait(false);

                if (response > 0)
                {
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.Package,
                        LogFrom = (int)AccountType.Administrator,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LocationId = model.LocationId,
                        LogDescription = $" {model.ModifiedByName} has added package {model.Name} on {DateTime.UtcNow.AddMinutes(330)}"
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                }

                return Ok(new GenericResponse
                {
                    Status = response switch
                    {
                        -1 => GenericStatus.Error,
                        -2 => GenericStatus.Info,
                        _ => GenericStatus.Success
                    },
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }
        
        /// <summary>
        /// The add icons async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] UpdateModel model,[FromHeader] LocationHeader header)
        {
            try
            {
                var response = await this.service.UpdateAsync(model).ConfigureAwait(false);

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Package,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $" {model.ModifiedByName} has updated package {model.Name} on {DateTime.UtcNow.AddMinutes(330)}",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);

                return Ok(new GenericResponse
                {
                    Status = response switch
                    {
                        -1 => GenericStatus.Error,
                        -2 => GenericStatus.Info,
                        _ => GenericStatus.Success
                    },
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }
        
        /// <summary>
        /// The add icons async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("delete-package-charge")]
        public async Task<ActionResult> DeletePackageChargeAsync([FromBody] DeleteModel model)
        {
            try
            {
                var response = await this.service.DeletePackageChargeAsync(model).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = response switch
                    {
                        -1 => GenericStatus.Error,
                        -2 => GenericStatus.Info,
                        _ => GenericStatus.Success
                    },
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }
        
        /// <summary>
        /// The add icons async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("delete")]
        public async Task<ActionResult> DeleteAsync([FromBody] MetaData.FilterModel model,[FromHeader]LocationHeader header)
        {
            try
            {
                var serviceid = await this.service.FindNameByPackageId(model.Id);

                var response = await this.service.DeleteAsync(model).ConfigureAwait(false);
                
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Package,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $" {model.ModifiedByName} has Deleted package {serviceid} on {DateTime.UtcNow.AddMinutes(330)}",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);

                return Ok(new GenericResponse
                {
                    Status = response switch
                    {
                        -1 => GenericStatus.Error,
                        -2 => GenericStatus.Info,
                        _ => GenericStatus.Success
                    },
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The add icons async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch-Products")]
        public async Task<ActionResult> FetchProductsAsync([FromBody] Product.FilterModel model)
        {
            try
            {
                var response = await this.service.FetchProductsAsync(model).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The add icons async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch-Product-masters")]
        public async Task<ActionResult> FetchProductMastersAsync()
        {
            try
            {
                var response = await this.service.FetchProductMastersAsync().ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The add icons async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch-surgeries")]
        public async Task<ActionResult> FetchSurgeriesAsync([FromBody] Surgery.FilterModel model)
        {
            try
            {
                var response = await this.service.FetchSurgeryKitsAsync(model).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The add icons async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch-surgery-masters")]
        public async Task<ActionResult> FetchSurgeryMastersAsync()
        {
            try
            {
                var response = await this.service.FetchSurgeryMastersAsync().ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }
    }
}